<?php

declare(strict_types=1);

namespace App\Controller;

use Dompdf\Dompdf;
use Dompdf\Options;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

/**
 * Citas Controller
 *
 * @property \App\Model\Table\CitasTable $Citas
 */
class CitasController extends AppController
{
public function exportarReporteExcel()
    {
        $fechaInicio  = $this->request->getQuery('fecha_inicio');
        $fechaFin     = $this->request->getQuery('fecha_fin');
        $doctorId     = $this->request->getQuery('doctor_id');
        $userId       = $this->request->getQuery('user_id');
        $estadoFiltro = $this->request->getQuery('estado');
        $campaaFiltro = $this->request->getQuery('campana_id');
        $tipoFecha    = $this->request->getQuery('tipo_fecha') ?? 'fecha_hora';

        if ($fechaInicio && $fechaFin) {
            try {
                $tz = new \DateTimeZone('America/Lima');
                $fechaInicio = new \DateTime($fechaInicio, $tz);
                $fechaFin    = new \DateTime($fechaFin, $tz);
                $fechaFin->modify('+1 day -1 second');
            } catch (\Exception $e) {
                $this->Flash->error('Formato de fechas inválido.');
                return $this->redirect(['action' => 'reportecitas']);
            }
        }

        // Consulta con relaciones y filtro por tipo de fecha
        $query = $this->Citas->find()
            ->contain([
                'Pacientes' => ['HistoriasClinicas'],
                'Doctores', 
                'Users', 
                'Campanas'
            ])
            ->where([
                "Citas.{$tipoFecha} >=" => $fechaInicio->format('Y-m-d H:i:s'),
                "Citas.{$tipoFecha} <=" => $fechaFin->format('Y-m-d H:i:s'),
            ]);

        if ($doctorId) {
            $query->where(['Citas.doctor_id' => $doctorId]);
        }
        if ($userId) {
            $query->where(['Citas.user_id' => $userId]);
        }
        if ($estadoFiltro) {
            $query->where(['estado' => $estadoFiltro]);
        }
        if ($campaaFiltro) {
            $query->where(['Campanas.id' => $campaaFiltro]);
        }

        $citas = $query->order(["Citas.{$tipoFecha}" => 'ASC'])->toArray();

        // Crear hoja Excel
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Encabezados (ordenado y consistente con las celdas)
        $headers = [
            'Estado',
            'Paciente',
            'DNI',
            'Doctor',
            $tipoFecha === 'created_at' ? 'Fecha de Creación' : 'Fecha de Cita',
            'Motivo',
            'Duración (min)',
            'Hora Llegada',
            'Inicio Consulta',
            'Fin Consulta',
            'Usuario',
            'Campaña'
        ];
        $sheet->fromArray($headers, null, 'A1');

        $totalPorEstado = [];
        $row = 2;

        foreach ($citas as $cita) {
            $estado = $cita->estado ?? 'Desconocido';
            if (!isset($totalPorEstado[$estado])) {
                $totalPorEstado[$estado] = 0;
            }
            $totalPorEstado[$estado]++;

            $paciente = !empty($cita->paciente)
                ? trim($cita->paciente->nombre . ' ' . $cita->paciente->apellido)
                : 'Paciente no encontrado';

            // Obtener DNI
            $dni = 'Sin DNI registrado';
            if (!empty($cita->paciente_id)) {
                $historiaClinica = $this->Citas->Pacientes->HistoriasClinicas->find()
                    ->select(['dni'])
                    ->where(['paciente_id' => $cita->paciente_id])
                    ->first();
                if ($historiaClinica && !empty($historiaClinica->dni)) {
                    $dni = $historiaClinica->dni;
                }
            }

            // Doctor
            $doctor = 'Doctor no asignado';
            if (!empty($cita->doctor)) {
                $doctor = trim($cita->doctor->nombre . ' ' . $cita->doctor->apellido);
            } elseif (!empty($cita->doctor_id)) {
                $doctorData = $this->Citas->Doctores->find()
                    ->select(['nombre', 'apellido'])
                    ->where(['id' => $cita->doctor_id])
                    ->first();
                if ($doctorData) {
                    $doctor = trim($doctorData->nombre . ' ' . $doctorData->apellido);
                }
            }

            $usuario = !empty($cita->user) ? $cita->user->username : 'Usuario no encontrado';

            $fechaCita    = $cita->fecha_hora instanceof \DateTimeInterface ? $cita->fecha_hora->format('Y-m-d H:i:s') : $cita->fecha_hora;
            $fechaCreada  = $cita->created_at instanceof \DateTimeInterface ? $cita->created_at->format('Y-m-d H:i:s') : $cita->created_at;
            $fechaMostrar = $tipoFecha === 'created_at' ? $fechaCreada : $fechaCita;

            $campana = !empty($cita->campana) ? $cita->campana->nombre : 'Sin campaña asignada';

            // NUEVOS CAMPOS (manejo de valores vacíos)
            $motivo   = !empty($cita->motivo) ? $cita->motivo : 'Sin motivo';
            $duracion = isset($cita->duracion_minutos) ? $cita->duracion_minutos : 15;

            // Horas: si están vacías mostrar "Sin hora marcada"
            $sinHoraTexto = 'Sin hora marcada';

            $horaLlegada = $sinHoraTexto;
            if (!empty($cita->hora_llegada)) {
                if ($cita->hora_llegada instanceof \DateTimeInterface) {
                    $horaLlegada = $cita->hora_llegada->format('H:i:s');
                } else {
                    $horaLlegada = (string)$cita->hora_llegada;
                }
            }

            $horaInicio = $sinHoraTexto;
            if (!empty($cita->hora_inicio_consulta)) {
                if ($cita->hora_inicio_consulta instanceof \DateTimeInterface) {
                    $horaInicio = $cita->hora_inicio_consulta->format('H:i:s');
                } else {
                    $horaInicio = (string)$cita->hora_inicio_consulta;
                }
            }

            $horaFin = $sinHoraTexto;
            if (!empty($cita->hora_fin_consulta)) {
                if ($cita->hora_fin_consulta instanceof \DateTimeInterface) {
                    $horaFin = $cita->hora_fin_consulta->format('H:i:s');
                } else {
                    $horaFin = (string)$cita->hora_fin_consulta;
                }
            }

            // ESCRIBIR EN EL Excel (A -> L)
            $sheet->setCellValue("A{$row}", $estado);
            $sheet->setCellValue("B{$row}", $paciente);
            $sheet->setCellValue("C{$row}", $dni);
            $sheet->setCellValue("D{$row}", $doctor);
            $sheet->setCellValue("E{$row}", $fechaMostrar);
            $sheet->setCellValue("F{$row}", $motivo);
            $sheet->setCellValue("G{$row}", $duracion);
            $sheet->setCellValue("H{$row}", $horaLlegada);
            $sheet->setCellValue("I{$row}", $horaInicio);
            $sheet->setCellValue("J{$row}", $horaFin);
            $sheet->setCellValue("K{$row}", $usuario);
            $sheet->setCellValue("L{$row}", $campana);

            $row++;
        }

        $row++;
        $sheet->setCellValue("A{$row}", "Totales por Estado:");
        $row++;

        foreach ($totalPorEstado as $estado => $cantidad) {
            $sheet->setCellValue("A{$row}", $estado);
            $sheet->setCellValue("B{$row}", $cantidad);
            $row++;
        }

        // Ajuste automático de columnas (ahora hasta L)
        foreach (range('A', 'L') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }


        $writer = new Xlsx($spreadsheet);
        $filename = 'reporte_citas_' . date('Ymd_His') . '.xlsx';
        $tmpFile = tempnam(sys_get_temp_dir(), 'xlsx');
        $writer->save($tmpFile);

               return $this->response->withFile($tmpFile, [
                'download' => true,
                'name' => $filename,
                'delete' => true,
            ]);
        }



    
   public function exportarReportePdf()
{
    $fechaInicio = $this->request->getQuery('fecha_inicio');
    $fechaFin = $this->request->getQuery('fecha_fin');
    $doctorId = $this->request->getQuery('doctor_id');
    $userId = $this->request->getQuery('user_id');
    $estadoFiltro = $this->request->getQuery('estado');
    $campaaFiltro = $this->request->getQuery('campana_id');
    $tipoFecha = $this->request->getQuery('tipo_fecha') ?? 'fecha_hora';
    $attachment = $this->request->getQuery('download') === 'true';

    $username = null;
    if ($userId) {
        $user = $this->Citas->Users->find()
            ->select(['username'])
            ->where(['id' => $userId])
            ->first();
        $username = $user ? $user->username : 'Usuario no encontrado';
    }

    if ($fechaInicio && $fechaFin) {
        try {
            $tz = new \DateTimeZone('America/Lima');
            $fechaInicio = new \DateTime($fechaInicio, $tz);
            $fechaFin = new \DateTime($fechaFin, $tz);
            $fechaFin->modify('+1 day -1 second');

            // Convertir a UTC para consulta en BD
            $fechaInicioUTC = clone $fechaInicio;
            $fechaFinUTC = clone $fechaFin;
            $fechaInicioUTC->setTimezone(new \DateTimeZone('UTC'));
            $fechaFinUTC->setTimezone(new \DateTimeZone('UTC'));
        } catch (\Exception $e) {
            $this->Flash->error('Formato de fechas inválido.');
            return $this->redirect(['action' => 'reportecitas']);
        }
    }

    $citasAgrupadas = [];
    $totalCitas = 0;

    if (isset($fechaInicioUTC) && isset($fechaFinUTC)) {
        $query = $this->Citas->find()
            ->contain(['Pacientes', 'Campanas', 'Users'])
            ->where([
                "Citas.{$tipoFecha} >=" => $fechaInicioUTC->format('Y-m-d H:i:s'),
                "Citas.{$tipoFecha} <=" => $fechaFinUTC->format('Y-m-d H:i:s'),
            ]);

        if ($doctorId) {
            $query->where(['Citas.doctor_id' => $doctorId]);
        }
        if ($userId) {
            $query->where(['Citas.user_id' => $userId]);
        }
        if ($estadoFiltro) {
            $query->where(['Citas.estado' => $estadoFiltro]);
        }
        if ($campaaFiltro) {
            $query->where(['Campanas.id' => $campaaFiltro]);
        }

        $citas = $query->order(["Citas.{$tipoFecha}" => 'ASC'])->toArray();

        $tzLima = new \DateTimeZone('America/Lima');

        foreach ($citas as $cita) {
            $doctor = $this->Citas->Doctores->find()
                ->select(['nombre', 'apellido'])
                ->where(['id' => $cita->doctor_id])
                ->first();
            $doctorNombreCompleto = $doctor
                ? $doctor->nombre . ' ' . $doctor->apellido
                : 'Doctor no asignado';

            $userNombre = $cita->user ? $cita->user->username : 'Usuario no encontrado';
            $estado = $cita->estado ?? 'Desconocido';

            if (!isset($citasAgrupadas[$estado])) {
                $citasAgrupadas[$estado] = [
                    'total' => 0,
                    'citas' => []
                ];
            }

            // ✅ Solo convertir zona horaria si el tipo es created_at (UTC → Lima)
            $fechaMostrar = 'No disponible';
            if ($tipoFecha === 'created_at' && !empty($cita->created_at)) {
                $dt = new \DateTime($cita->created_at->format('Y-m-d H:i:s'), new \DateTimeZone('UTC'));
                $dt->setTimezone($tzLima);
                $fechaMostrar = $dt->format('Y-m-d H:i:s');
            } elseif ($tipoFecha === 'fecha_hora' && !empty($cita->fecha_hora)) {
                $fechaMostrar = $cita->fecha_hora->format('Y-m-d H:i:s'); // ya está en Lima
            }

            $pacienteNombre = $cita->paciente
                ? $cita->paciente->nombre . ' ' . $cita->paciente->apellido
                : 'Paciente no encontrado';

            $campanaNombre = $cita->campana
                ? $cita->campana->nombre
                : 'Sin campaña asignada';

            // Procesar campos adicionales similar al Excel
            $motivo = !empty($cita->motivo) ? $cita->motivo : 'Sin motivo';
            $duracion = isset($cita->duracion_minutos) ? $cita->duracion_minutos : 15;

            // Horas: si están vacías mostrar "Sin hora marcada"
            $sinHoraTexto = 'Sin hora marcada';

            $horaLlegada = $sinHoraTexto;
            if (!empty($cita->hora_llegada)) {
                if ($cita->hora_llegada instanceof \DateTimeInterface) {
                    $horaLlegada = $cita->hora_llegada->format('H:i:s');
                } else {
                    $horaLlegada = (string)$cita->hora_llegada;
                }
            }

            $horaInicio = $sinHoraTexto;
            if (!empty($cita->hora_inicio_consulta)) {
                if ($cita->hora_inicio_consulta instanceof \DateTimeInterface) {
                    $horaInicio = $cita->hora_inicio_consulta->format('H:i:s');
                } else {
                    $horaInicio = (string)$cita->hora_inicio_consulta;
                }
            }

            $horaFin = $sinHoraTexto;
            if (!empty($cita->hora_fin_consulta)) {
                if ($cita->hora_fin_consulta instanceof \DateTimeInterface) {
                    $horaFin = $cita->hora_fin_consulta->format('H:i:s');
                } else {
                    $horaFin = (string)$cita->hora_fin_consulta;
                }
            }

            $citasAgrupadas[$estado]['total']++;
            $citasAgrupadas[$estado]['citas'][] = [
                'paciente' => $pacienteNombre,
                'doctor' => $doctorNombreCompleto,
                'fecha_mostrar' => $fechaMostrar,
                'usuario' => $userNombre,
                'campana' => $campanaNombre,
                'motivo' => $motivo,
                'duracion_minutos' => $duracion,
                'hora_llegada' => $horaLlegada,
                'hora_inicio_consulta' => $horaInicio,
                'hora_fin_consulta' => $horaFin,
            ];
        }

        $totalCitas = count($citas);
    }

    $this->viewBuilder()->disableAutoLayout();
    $this->set(compact(
        'fechaInicio',
        'fechaFin',
        'doctorId',
        'userId',
        'estadoFiltro',
        'totalCitas',
        'citasAgrupadas',
        'username',
        'campaaFiltro',
        'tipoFecha'
    ));

    $html = $this->render('reportecitas_pdf')->getBody()->__toString();

    $options = new \Dompdf\Options();
    $options->set('defaultFont', 'DejaVu Sans');
    $dompdf = new \Dompdf\Dompdf($options);
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4', 'portrait');
    $dompdf->render();

    if ($attachment) {
        $dompdf->stream("reporte_citas.pdf", ["Attachment" => true]);
    } else {
        $this->response = $this->response->withType('application/pdf');
        $this->response = $this->response->withStringBody($dompdf->output());
        return $this->response;
    }
}




    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */

    public function index()
    {
        // Obtener la identidad del usuario autenticado
        $usuario = $this->Authentication->getIdentity();

        if (!$usuario) {
            // Redirigir al login si no hay usuario autenticado
            return $this->redirect(['controller' => 'Usuarios', 'action' => 'login']);
        }

        // Obtener el doctor de la URL si existe
        $doctorId = $this->request->getQuery('doctor_id');
        $fecha = $this->request->getQuery('fecha') ?? date('Y-m-d');

        // Si el usuario es doctor (rol = 3) y no hay un doctor en la URL, asignar automáticamente su doctor_id
        if (!$doctorId && $usuario->rol == 3 && !empty($usuario->doctor_id)) {
            $doctorId = $usuario->doctor_id;
        }

        // Consulta base de las citas
        $query = $this->Citas->find()
            ->contain(['Doctores', 'Pacientes']);


        // Filtrar por doctor si hay un doctor seleccionado (excepto para admins)
        if ($doctorId) {
            $query->where(['Citas.doctor_id' => $doctorId]);
        }

        // Filtrar por fecha seleccionada
        if ($fecha) {
            $query->where(['DATE(Citas.fecha_hora)' => $fecha]);
        }

        // Obtener resultados
        $citas = $query->all();

        // Obtener listas para selectores
        $doctores = $this->Citas->Doctores->find(
            'list',
            keyField: 'id',
            valueField: function ($doctor) {
                return $doctor->nombre . ' ' . $doctor->apellido;
            }
        )->toArray();
        
        asort($doctores);

        // Pasar datos a la vista
        $this->set(compact('citas', 'doctores', 'doctorId', 'fecha', 'usuario'));
    }


    /**
     * View method
     *
     * @param string|null $id Cita id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $roles = [1, 2, 3]; // Roles permitidos
        if (!in_array($this->request->getAttribute('identity')->rol, $roles)) {
            $this->Flash->error(__('No tienes permisos para acceder a esta sección'));
            return $this->redirect($this->referer());
        }
        $cita = $this->Citas->get(
            $id,
            contain: ['Doctores', 'Campanas', 'Pacientes','CitasTratamientos'=> ['Tratamientos']]
        );

        $this->set(compact('cita'));
        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */

    public function add()
    {
        $roles = [1, 2];
        if (!in_array($this->request->getAttribute('identity')->rol, $roles)) {
            $this->Flash->error(__('No tienes permisos para acceder a esta sección'));
            return $this->redirect($this->referer());
        }
        $cita = $this->Citas->newEmptyEntity();
        $doctorSeleccionado = null; // Inicializar variable

        // Prellenar datos si se reciben como parámetros en la URL
        if ($this->request->is('get')) {
            if ($this->request->getQuery('fecha_hora')) {
                $cita->fecha_hora = $this->request->getQuery('fecha_hora');
            }
            if ($this->request->getQuery('doctor_id')) {
                $cita->doctor_id = $this->request->getQuery('doctor_id');
                $doctorSeleccionado = $this->request->getQuery('doctor_id'); // Guardar el doctor seleccionado
            }
        }

        if ($this->request->is('post')) {
            $data = $this->request->getData();

            // Si viene información de un nuevo paciente
            if (!empty($data['nuevo_paciente']['nombre']) && !empty($data['nuevo_paciente']['apellido'])) {
                $pacienteData = $data['nuevo_paciente'];
                $paciente = $this->Citas->Pacientes->newEntity($pacienteData);

                if ($this->Citas->Pacientes->save($paciente)) {
                    $data['paciente_id'] = $paciente->id; // Asignar el nuevo paciente a la cita
                } else {
                    $this->Flash->error(__('No se pudo crear el nuevo paciente. Por favor, intente de nuevo.'));
                }
            }

            // Asignar el usuario logueado a la cita
            $data['user_id'] = $this->request->getAttribute('identity')->id;

            $cita = $this->Citas->patchEntity($cita, $data, [
                'associated' => ['CitasTratamientos', 'Campanas']
            ]);

            // Intentar guardar la cita
            if ($this->Citas->save($cita)) {
                if (!empty($data['citas_tratamientos']) && is_array($data['citas_tratamientos'])) {
                    $citasTratamientosTable = $this->fetchTable('CitasTratamientos');

                    foreach ($data['citas_tratamientos'] as $recetaId => $recetaData) {
                        if (is_array($recetaData)) {
                            $citasTratamientos = $citasTratamientosTable->newEntity([
                                'cita_id' => $cita->id,
                                'tratamiento_id' => $recetaId,
                            ]);
                            $citasTratamientosTable->save($citasTratamientos);
                        }
                    }
                }
                $this->Flash->success(__('La cita se guardó correctamente.'));
                return $this->redirect([
                    'action' => 'index',
                    '?' => [
                        'doctor_id' => $cita->doctor_id,
                        'fecha' => $cita->fecha_hora->format('Y-m-d'),
                    ]
                ]);
            }

            $this->Flash->error(__('No se pudo guardar la cita. Por favor, inténtelo de nuevo.'));
        }

        $doctores = $this->Citas->Doctores->find('list')->all();
        $pacientes = $this->Citas->Pacientes->find('list')->all();
        $campanas = $this->Citas->Campanas->find('list')->toArray();
        // ✅ Obtener lista de tratamientos
        $tratamientos = $this->Citas->Tratamientos->find('list', keyField: 'id', valueField: 'nombre')->all();
        asort($campanas, SORT_NATURAL | SORT_FLAG_CASE);
        // Pasar la variable doctorSeleccionado a la vista
        $this->set(compact('cita', 'doctores', 'pacientes', 'doctorSeleccionado', 'tratamientos','campanas'));

        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }

    /**
     * Edit method
     *
     * @param string|null $id Cita id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        // Verifica si el usuario tiene permiso para acceder a esta sección
        $roles = [1, 2];
        if (!in_array($this->request->getAttribute('identity')->rol, $roles)) {
            $this->Flash->error(__('No tienes permisos para acceder a esta sección'));
            return $this->redirect($this->referer());
        }

        // Verifica que el ID de la cita sea válido
        if (!$id) {
            $this->log('El ID no fue recibido en el controlador.', 'error');
            if ($this->request->is('ajax')) {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => false, 'message' => __('ID de cita no válido.')]));
            }
            $this->Flash->error(__('ID de cita no válido.'));
            return $this->redirect(['action' => 'index']);
        }

        // Intenta obtener la cita
        try {
            $cita = $this->Citas->get($id, 
                contain : ['Campanas']
            );
        } catch (\Exception $e) {
            $this->log('Error al obtener la cita: ' . $e->getMessage(), 'error');
            if ($this->request->is('ajax')) {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => false, 'message' => __('La cita no existe.')]));
            }
            $this->Flash->error(__('La cita no existe.'));
            return $this->redirect(['action' => 'index']);
        }

        // Procesa el formulario cuando es enviado (PATCH, POST, PUT)
        if ($this->request->is(['patch', 'post', 'put'])) {
            // Obtener los datos del formulario correctamente usando getData()
            $requestData = $this->request->getData();

            // **Combinamos la fecha y la hora solo si se han proporcionado ambos campos**
            if (isset($requestData['fecha']) && isset($requestData['hora'])) {
                $fecha = $requestData['fecha'];
                $hora = $requestData['hora'];

                // Crear una fecha y hora combinada en formato "Y-m-d H:i:s"
                $fechaHora = $fecha . ' ' . $hora;
                $requestData['fecha_hora'] = $fechaHora; // Asignamos el valor combinado a `fecha_hora`
            }

            // Aplicar los cambios en la entidad Cita
            $cita = $this->Citas->patchEntity($cita, $requestData);

            // Si se guardó correctamente
            if ($this->Citas->save($cita)) {
                if (!empty($requestData['citas_tratamientos'])) {
                    $citasTratamientosTable = $this->fetchTable('CitasTratamientos');

                    // Eliminar los tratamientos anteriores asociados a esta cita
                    $citasTratamientosTable->deleteAll(['cita_id' => $cita->id]);

                    // Insertar los nuevos tratamientos
                    foreach ($requestData['citas_tratamientos'] as $t) {
                        $nuevaAsociacion = $citasTratamientosTable->newEmptyEntity();
                        $nuevaAsociacion = $citasTratamientosTable->patchEntity($nuevaAsociacion, [
                            'cita_id' => $cita->id,
                            'tratamiento_id' => $t['tratamiento_id'] ?? null,
                            'cantidad' => $t['cantidad'] ?? 1,
                            'total' => $t['total'] ?? 0
                        ]);
                        $citasTratamientosTable->save($nuevaAsociacion);
                    }
                }

                // Si es una solicitud AJAX
                if ($this->request->is('ajax')) {
                    return $this->response->withType('application/json')
                        ->withStringBody(json_encode([
                            'success' => true,
                            'message' => __('Cita actualizada correctamente.'),
                            'doctor_id' => $cita->doctor_id,
                            'fecha_hora' => $cita->fecha_hora->format('Y-m-d')
                        ]));
                }

                // Si no es AJAX, muestra mensaje y redirige
                $this->Flash->success(__('Cita actualizada correctamente.'));

                // Redirigir al índice manteniendo el odontólogo y la fecha seleccionados
                return $this->redirect([
                    'action' => 'index',
                    '?' => [
                        'doctor_id' => $cita->doctor_id,
                        'fecha' => $cita->fecha_hora->format('Y-m-d'),
                    ]
                ]);
            }

            // Si no se pudo guardar, mostrar error
            if ($this->request->is('ajax')) {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => false, 'message' => __('No se pudo actualizar la cita.')]));
            }
            $this->Flash->error(__('No se pudo actualizar la cita. Por favor, inténtelo de nuevo.'));
        }

        // Obtener la lista de doctores y pacientes para mostrarlos en el formulario
        $doctores = $this->Citas->Doctores->find('list')->toArray();
        $pacientes = $this->Citas->Pacientes->find('list')->toArray();
        // Cargar los tratamientos disponibles
        $tratamientos = $this->Citas->Tratamientos->find('list')->toArray();
        $campanas = $this->Citas->Campanas->find('list')->toArray();
        // Obtener tratamientos ya asignados a la cita
        $citasTratamientosTable = $this->fetchTable('CitasTratamientos');
        $tratamientosAsociados = $citasTratamientosTable->find()
            ->where(['cita_id' => $id])
            ->contain(['Tratamientos'])
            ->all()
            ->toArray();

        // Asignar la cita, doctores y pacientes a la vista
        $this->set(compact('cita', 'doctores', 'pacientes', 'tratamientos', 'tratamientosAsociados','campanas'));

        // Si es una solicitud AJAX, usar un layout diferente
        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }


    /**
     * Delete method
     *
     * @param string|null $id Cita id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $cita = $this->Citas->get($id);

        if ($this->Citas->delete($cita)) {
            $this->Flash->success(__('The cita has been deleted.'));
        } else {
            $this->Flash->error(__('The cita could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function fetchCitas()
    {
        $this->request->allowMethod(['get']);

        // Obtener los parámetros de la consulta
        $fecha = $this->request->getQuery('fecha');
        $doctorId = $this->request->getQuery('doctor_id');

        // Validar que se hayan enviado los parámetros
        if (!$fecha || !$doctorId) {
            return $this->response->withType('application/json')
                ->withStringBody(json_encode(['error' => 'Faltan parámetros necesarios']));
        }

        // Calcular el rango de fechas para la semana seleccionada
        $fechaInicio = new \DateTime($fecha, new \DateTimeZone('America/Lima'));
        $fechaFin = clone $fechaInicio;
        $fechaFin->modify('+6 days');

        // Buscar el nombre completo del doctor por el doctor_id
        $doctor = $this->Citas->Doctores->find()
            ->select(['nombre', 'apellido'])
            ->where(['id' => $doctorId])
            ->first();

        // Asignar nombre del doctor
        $doctorNombreCompleto = $doctor
            ? $doctor->nombre . ' ' . $doctor->apellido
            : 'Doctor no asignado';

        // Buscar las citas en el rango de fechas y para el doctor seleccionado
        $citas = $this->Citas->find()
            ->select([
                'id',
                'doctor_id',
                'fecha_hora',
                'paciente_id',
                'estado',
                'motivo',
                'hora_llegada',
                'duracion_minutos',
                'Pacientes.nombre',
                'Pacientes.apellido',
                'Pacientes.telefono_celular',
            ])
            ->contain(['Pacientes']) // Solo contener Pacientes
            ->where([
                'fecha_hora >=' => $fechaInicio->format('Y-m-d 00:00:00'),
                'fecha_hora <=' => $fechaFin->format('Y-m-d 23:59:59'),
                'doctor_id' => $doctorId
            ])
            ->order(['fecha_hora' => 'ASC'])
            ->toArray();

         //Diferenciar citas con historia: Obtener todos los paciente_ids de las citas
            $pacienteIds = array_column($citas, 'paciente_id');
            
            // Hacer una consulta directa a historias clínicas para estos pacientes
            $historiasClinicas = [];
            if (!empty($pacienteIds)) {
                $historias = $this->Citas->Pacientes->HistoriasClinicas->find()
                    ->select(['paciente_id', 'dni'])
                    ->where(['paciente_id IN' => $pacienteIds])
                    ->toArray();
                
                // Crear un array asociativo para acceso rápido
                foreach ($historias as $historia) {
                    $historiasClinicas[$historia->paciente_id] = $historia;
                }
            }

            // Formatear las citas para devolverlas en JSON
            $citasFormateadas = array_map(function ($cita) use ($doctorNombreCompleto, $historiasClinicas) {
            $citaHoraLima = $cita->fecha_hora->setTimezone(new \DateTimeZone('America/Lima'));
             // Verificar si este paciente tiene historia clínica con DNI
            $tieneHistoriaClinicaDni = false;
            if (isset($historiasClinicas[$cita->paciente_id])) {
                $historia = $historiasClinicas[$cita->paciente_id];
                $tieneHistoriaClinicaDni = !empty($historia->dni);
            }
            return [
                'id' => $cita->id,
                'doctor_id' => $cita->doctor_id,
                'fecha_hora' => $citaHoraLima->format('Y-m-d H:i:s'),
                'duracion_minutos' => $cita->duracion_minutos,
                'paciente_id' => $cita->paciente_id,
                'paciente' => $cita->paciente->nombre . ' ' . $cita->paciente->apellido,
                'telefono_celular' => $cita->paciente->telefono_celular,
                'estado' => $cita->estado,
                'hora_llegada' =>  $cita->hora_llegada,
                'doctor' => $doctorNombreCompleto,
                'tiene_historia_clinica_dni' => $tieneHistoriaClinicaDni,
                'motivo' => $cita->motivo ?? '',
            ];
        }, $citas);

        return $this->response->withType('application/json')
            ->withStringBody(json_encode($citasFormateadas));
    }

    public function changeStatus()
    {
        $this->request->allowMethod(['post']);

        $id = $this->request->getData('id');
        $estado = $this->request->getData('estado');

        if (!$id || !$estado) {
            return $this->response->withType('application/json')
                ->withStringBody(json_encode(['success' => false, 'message' => 'Parámetros inválidos.']));
        }

        try {
            $cita = $this->Citas->get($id);
            $cita->estado = $estado;

            if ($this->Citas->save($cita)) {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => true, 'message' => 'Estado actualizado correctamente.']));
            } else {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => false, 'message' => 'No se pudo actualizar el estado.']));
            }
        } catch (\Exception $e) {
            return $this->response->withType('application/json')
                ->withStringBody(json_encode(['success' => false, 'message' => 'Error al procesar la solicitud.']));
        }
    }

        public function reportecitas()
    {
        // Obtener los parámetros del rango de fechas y doctor desde el formulario
        $fechaInicio = $this->request->getQuery('fecha_inicio');
        $fechaFin = $this->request->getQuery('fecha_fin');
        $doctorId = $this->request->getQuery('doctor_id');
        $userId = $this->request->getQuery('user_id');
        $campanaId = $this->request->getQuery('campana_id'); // Nuevo parámetro para campaña

        if ($fechaInicio && $fechaFin) {
            try {
                $fechaInicio = new \DateTime($fechaInicio, new \DateTimeZone('America/Lima'));
                $fechaFin = new \DateTime($fechaFin, new \DateTimeZone('America/Lima'));
                $fechaFin->modify('+1 day -1 second'); // Ajustar para incluir todo el día final
            } catch (\Exception $e) {
                $this->Flash->error('Formato de fechas inválido.');
                $fechaInicio = null;
                $fechaFin = null;
            }
        }

        $citasAgrupadas = [];
        $totalCitas = 0;

        if ($fechaInicio && $fechaFin) {
            $query = $this->Citas->find()
                ->select([
                    'id',
                    'doctor_id',
                    'fecha_hora',
                    'estado',
                    'Pacientes.nombre',
                    'Pacientes.apellido',
                    'user_id',
                    'Campanas.nombre' // Asegurarse de obtener el nombre de la campaña
                ])
                ->contain(['Pacientes', 'Campañas']) // Relacionar solo Pacientes
                ->where([
                    'fecha_hora >=' => $fechaInicio->format('Y-m-d H:i:s'),
                    'fecha_hora <=' => $fechaFin->format('Y-m-d H:i:s')
                ]);

            // Filtrar por doctor solo si se selecciona uno
            if ($doctorId) {
                $query->where(['Citas.doctor_id' => $doctorId]);
            }

            // Filtrar por user_id si se selecciona uno
            if ($userId) {
                $query->where(['Citas.user_id' => $userId]);
            }

            // Filtrar por campaña si se selecciona una
            if ($campanaId) {
                $query->where(['Campanas.id' => $campanaId]);
            }
            $citas = $query->order(['fecha_hora' => 'ASC'])->toArray();

            foreach ($citas as $cita) {
                // Consultar el nombre completo del doctor
                $doctor = $this->Citas->Doctores->find()
                    ->select(['nombre', 'apellido'])
                    ->where(['id' => $cita->doctor_id])
                    ->first();
                $doctorNombreCompleto = $doctor
                    ? $doctor->nombre . ' ' . $doctor->apellido
                    : 'Doctor no asignado';

                // Agrupar los datos por estado
                $estado = $cita->estado ?? 'Desconocido';
                if (!isset($citasAgrupadas[$estado])) {
                    $citasAgrupadas[$estado] = [
                        'total' => 0,
                        'citas' => []
                    ];
                }
                $citasAgrupadas[$estado]['total']++;
                $citasAgrupadas[$estado]['citas'][] = [
                    'paciente' => $cita->paciente->nombre . ' ' . $cita->paciente->apellido,
                    'doctor' => $doctorNombreCompleto,
                    'fecha_hora' => $cita->fecha_hora->format('Y-m-d H:i:s'),
                    'campana' => $cita->campana ? $cita->campana->nombre : 'Sin campaña',
                ];
            }

            $totalCitas = count($citas);
        }

        // Obtener la lista de doctores para el filtro
        $doctores = $this->Citas->Doctores->find(
            'list',
            keyField: 'id',
            valueField: function ($doctor) {
                return $doctor->nombre . ' ' . $doctor->apellido;
            }
        )->toArray();
        // Obtener la lista de usuarios para el filtro (asegúrate de tener esta consulta)
        $usuarios = $this->Citas->Users->find(
            'list',
            keyField: 'id',
            valueField: 'username' // O el campo que prefieras
        )->toArray();
        $estados = [
            'pendiente' => 'Pendiente',
            'confirmado' => 'Confirmado',
            'en_consultorio' => 'En Consultorio',
            'finalizado' => 'Finalizado',
            'cancelado' => 'Cancelado'
        ];
        // Obtener la lista de campañas para el filtro
        $campanas = $this->Citas->Campanas->find('list', keyField: 'id', valueField: 'nombre')->toArray();
        $this->set(compact('fechaInicio', 'fechaFin', 'doctorId', 'totalCitas', 'citasAgrupadas', 'doctores', 'usuarios', 'estados', 'campanas'));
    }


    // public function verificarDisponibilidad()
    // {
    //     $this->request->allowMethod(['get']);

    //     // Obtener los parámetros de la URL
    //     $fechaHora = $this->request->getQuery('fecha_hora');
    //     $doctorId = $this->request->getQuery('doctor_id');
    //     $citaId = $this->request->getQuery('cita_id'); // El ID de la cita que estamos editando

    //     // Convertir la fecha a objeto DateTime
    //     $inicioNuevaCita = new \DateTime($fechaHora);

    //     // Duración en minutos (por defecto 15 minutos si no se envía)
    //     $duracionMinutos = (int) $this->request->getQuery('duracion_minutos', 15);
    //     $finNuevaCita = (clone $inicioNuevaCita)->modify("+{$duracionMinutos} minutes");

    //     // Consulta para encontrar citas que se solapen con la nueva
    //     $citaExistente = $this->Citas->find()
    //         ->where([
    //             'doctor_id' => $doctorId,
    //             'OR' => [
    //                 [
    //                     'fecha_hora <' => $finNuevaCita->format('Y-m-d H:i:s'),
    //                     $this->Citas->query()->newExpr("DATE_ADD(fecha_hora, INTERVAL duracion_minutos MINUTE) > '" . $inicioNuevaCita->format('Y-m-d H:i:s') . "'")
    //                 ]
    //             ]
    //         ]);

    //     // Si estamos editando una cita, excluir la cita actual (si se proporcionó un citaId)
    //     if ($citaId !== null) {
    //         $citaExistente->andWhere(['Citas.id !=' => $citaId]);
    //     }

    //     // Obtener el primer resultado
    //     $citaExistente = $citaExistente->first();

    //     // Si se encontró una cita existente que se solapa, la hora está ocupada
    //     $disponible = $citaExistente ? false : true;

    //     // Devolver respuesta en formato JSON
    //     return $this->response->withType('application/json')
    //         ->withStringBody(json_encode(['disponible' => $disponible]));
    // }

    public function verificarDisponibilidad()
    {
        $this->request->allowMethod(['get']);

        // Obtener los parámetros de la URL
        $fechaHora = $this->request->getQuery('fecha_hora');
        $doctorId = $this->request->getQuery('doctor_id');
        $citaId = $this->request->getQuery('cita_id'); // El ID de la cita que estamos editando

        // Convertir la fecha a objeto DateTime
        $inicioNuevaCita = new \DateTime($fechaHora);

        // Duración en minutos (por defecto 15 minutos si no se envía)
        $duracionMinutos = (int) $this->request->getQuery('duracion_minutos', 15);
        $finNuevaCita = (clone $inicioNuevaCita)->modify("+{$duracionMinutos} minutes");

        // Consulta para encontrar citas que se solapen con la nueva
        $citaExistente = $this->Citas->find()
            ->where([
                'doctor_id' => $doctorId,
                'estado !=' => 'cancelado', // ignoramos las citas canceladas en la verificacion
                'OR' => [
                    [
                        'fecha_hora <' => $finNuevaCita->format('Y-m-d H:i:s'),
                        $this->Citas->query()->newExpr("DATE_ADD(fecha_hora, INTERVAL duracion_minutos MINUTE) > '" . $inicioNuevaCita->format('Y-m-d H:i:s') . "'")
                    ]
                ]
            ]);

        // Si estamos editando una cita, excluir la cita actual (si se proporcionó un citaId)
        if ($citaId !== null) {
            $citaExistente->andWhere(['Citas.id !=' => $citaId]);
        }

        // Obtener el primer resultado
        $citaExistente = $citaExistente->first();

        // Si se encontró una cita existente que se solapa, la hora está ocupada
        $disponible = $citaExistente ? false : true;

        // Devolver respuesta en formato JSON
        return $this->response->withType('application/json')
            ->withStringBody(json_encode(['disponible' => $disponible]));
    }
    public function citaDiaria()
    {
        // Obtener la lista de doctores
        $doctores = $this->Citas->Doctores->find(
            'list',
            keyField: 'id',
            valueField: function ($row) {
                return $row->nombre . ' ' . $row->apellido;
            }
        )->toArray();
        
        asort($doctores);
        
        $coloresDoctores = [
            "#e6194b", "#3cb44b", "#ffe119", "#4363d8", "#f58231", "#911eb4", "#46f0f0", "#f032e6",
            "#bcf60c", "#fabebe", "#008080", "#e6beff", "#9a6324", "#fffac8", "#800000", "#aaffc3",
            "#808000", "#ffd8b1", "#000075", "#808080", "#ffffff", "#000000", "#1f77b4", "#ff7f0e",
            "#2ca02c", "#d62728", "#9467bd", "#8c564b", "#e377c2", "#7f7f7f"
        ];

        // Obtener doctor seleccionado (si lo hay)
        $doctorId = $this->request->getQuery('doctor_id');
        
        $usuario = $this->Authentication->getIdentity();

        // Si el usuario es doctor (rol = 3) y no hay un doctor en la URL, asignar automáticamente su doctor_id
        if (!$doctorId && $usuario->rol == 3 && !empty($usuario->doctor_id)) {
            $doctorId = $usuario->doctor_id;
        }

        // Obtener la lista de tratamientos
        $tratamientos = $this->Citas->CitasTratamientos->Tratamientos->find('list', keyField: 'id', valueField: 'nombre')->toArray();

        // Validar que el parámetro 'fecha' esté presente
        $fecha = $this->request->getQuery('fecha', date('Y-m-d')); // Si no hay fecha, usar hoy

        // Calcular el rango de fechas para hoy
        $fechaInicio = new \DateTime($fecha . ' 00:00:00', new \DateTimeZone('America/Lima'));
        $fechaFin = clone $fechaInicio;
        $fechaFin->modify('+1 day')->modify('-1 second'); // Último segundo del día

        // Construir la consulta
        $query = $this->Citas->find()
            ->contain(['Pacientes', 'Doctores', 'CitasTratamientos.Tratamientos']) // Asegúrate de cargar la relación con Tratamientos
            ->where([
                'fecha_hora >=' => $fechaInicio->format('Y-m-d H:i:s'),
                'fecha_hora <=' => $fechaFin->format('Y-m-d H:i:s')
            ]);
        if (!empty($doctorId)) {
            $query->where(['Citas.doctor_id' => $doctorId]);
        }
        // Obtener las citas
        $citas = $query->order(['fecha_hora' => 'ASC'])->toArray();


        // Pasar las variables a la vista
        $this->set(compact('citas', 'doctores', 'doctorId', 'tratamientos','usuario', 'coloresDoctores'));
    }


    public function registrarInicio($id)
    {
        $cita = $this->Citas->get($id);
        $cita->hora_inicio_consulta = date('H:i:s');

        if ($this->Citas->save($cita)) {
            $this->Flash->success('Inicio de consulta registrado.');
        } else {
            $this->Flash->error('No se pudo registrar.');
        }

        // Redirigir al índice manteniendo el odontólogo y la fecha seleccionados
        return $this->redirect([
            'action' => 'citaDiaria',
            '?' => [
                'doctor_id' => $cita->doctor_id,
            ]
        ]);
    }
    public function finalizarConsulta($id)
    {
        $cita = $this->Citas->get($id);
        $cita->hora_fin_consulta = date('H:i:s');
        $cita->estado = 'finalizado';

        if ($this->Citas->save($cita)) {
            $this->Flash->success('Consulta finalizada.');
        } else {
            $this->Flash->error('No se pudo finalizar.');
        }

        return $this->redirect([
            'action' => 'citaDiaria',
            '?' => [
                'doctor_id' => $cita->doctor_id,
            ]
        ]);
    }

    public function marcarHoraLlegada()
    {
        $this->request->allowMethod(['post']);
        $citaId = $this->request->getData('cita_id');
        $horaLlegada = $this->request->getData('hora_llegada');

        $cita = $this->Citas->get($citaId);
        $cita->hora_llegada = $horaLlegada;

        if ($this->Citas->save($cita)) {
            $this->Flash->success('Hora de llegada registrada.');
        } else {
            $this->Flash->error('No se pudo registrar la hora de llegada.');
        }

        return $this->redirect(['action' => 'citaDiaria']);
    }

    // src/Controller/CitasController.php
    // src/Controller/CitasController.php


    public function actualizarHora()
    {
        $this->request->allowMethod(['post']);

        // Obtener los datos del formulario
        $citaId = $this->request->getData('cita_id');
        $nuevaFecha = $this->request->getData('nueva_fecha'); // Fecha proporcionada
        $nuevaHora = $this->request->getData('nueva_hora'); // Hora proporcionada

        // Combinar la nueva fecha con la nueva hora y agregar segundos: 'YYYY-MM-DD HH:MM:SS'
        $fechaHoraStr = $nuevaFecha . ' ' . $nuevaHora . ':00'; // Agregar los segundos (00)

        // Intentar crear un objeto DateTime para validar la fecha y hora
        try {
            $fechaHora = new \DateTime($fechaHoraStr);  // Crear un objeto DateTime con la fecha y hora proporcionada

            // Obtener la cita
            $cita = $this->Citas->get($citaId);
            $cita->fecha_hora = $fechaHora->format('Y-m-d H:i:s'); // Formatear correctamente la fecha y hora

            // Guardar la cita con la nueva fecha y hora
            if ($this->Citas->save($cita)) {
                $this->Flash->success('La hora de la cita ha sido actualizada correctamente.');
            } else {
                $this->Flash->error('No se pudo actualizar la hora de la cita.');
            }
        } catch (\Exception $e) {
            // Si no se pudo crear el objeto DateTime
            $this->Flash->error('Fecha y hora inválidas.');
        }

        // Redirigir a la vista correspondiente
        return $this->redirect(['action' => 'index']);
    }
    
     public function restablecerCita()
    {
        $this->request->allowMethod(['post']);

        $citaId = $this->request->getData('cita_id');

        if (!$citaId) {
            return $this->response->withType('application/json')
                ->withStringBody(json_encode(['success' => false, 'message' => 'ID de cita no proporcionado.']));
        }

        try {
            $cita = $this->Citas->get($citaId);
            $cita->estado = 'pendiente';
            $cita->hora_llegada = null;

            if ($this->Citas->save($cita)) {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode(['success' => true, 'message' => 'Cita restablecida correctamente.']));
            } else {
                return $this->response->withType('application/json')
                    ->withStringBody(json_encode([
                        'success' => false,
                        'message' => 'No se pudo guardar la cita.',
                        'errors' => $cita->getErrors()
                    ]));
            }
        } catch (\Exception $e) {
            return $this->response->withType('application/json')
                ->withStringBody(json_encode([
                    'success' => false,
                    'message' => 'Error del servidor.',
                    'exception' => $e->getMessage()
                ]));
        }
    }
    
    public function actualizarTablaCitas() 
    {
        // Verificar que sea una petición AJAX
        if (!$this->request->is('ajax')) {
            throw new NotFoundException('Acceso no permitido');
        }
        
        $coloresDoctores = [
            "#e6194b", "#3cb44b", "#ffe119", "#4363d8", "#f58231", "#911eb4", "#46f0f0", "#f032e6",
            "#bcf60c", "#fabebe", "#008080", "#e6beff", "#9a6324", "#fffac8", "#800000", "#aaffc3",
            "#808000", "#ffd8b1", "#000075", "#808080", "#ffffff", "#000000", "#1f77b4", "#ff7f0e",
            "#2ca02c", "#d62728", "#9467bd", "#8c564b", "#e377c2", "#7f7f7f"
        ];
        
        // $this->viewBuilder()->setLayout(null);
        $this->viewBuilder()->setLayout('ajax');
    
        
        // Obtener usuario actual desde la sesión
        $session = $this->request->getSession();
        $usuario = $session->read('Auth.User') ?: $session->read('Auth');
        
        // Si no hay usuario en sesión, usar valores por defecto
        if (!$usuario) {
            $usuario = ['rol' => 1, 'id' => null]; // Ajusta según tu estructura
        }
        
        $doctorId = $this->request->getQuery('doctor_id');
        
        // Reutilizar la misma lógica que en tu método index o cita_diaria
        $query = $this->Citas->find()
            ->contain(['Pacientes', 'CitasTratamientos.Tratamientos', 'Doctores'])
            ->where(['DATE(Citas.fecha_hora)' => date('Y-m-d')])
            ->order(['Citas.fecha_hora' => 'ASC']);
        
        // Filtrar por doctor si está seleccionado
        if (!empty($doctorId)) {
            $query->where(['Citas.doctor_id' => $doctorId]);
        } elseif (isset($usuario['rol']) && $usuario['rol'] == 3) { // Si es doctor, solo sus citas
            $query->where(['Citas.doctor_id' => $usuario['id']]);
        }
        
        $citas = $query->toArray();
        
        // Preparar doctores para el filtro (si es necesario)
        if (!isset($usuario['rol']) || $usuario['rol'] != 3) {
            $doctores = $this->Citas->Doctores->find('list', keyField: 'id', valueField: function ($doctor) {
                return $doctor->nombre . ' ' . $doctor->apellido;
            })->toArray();
        } else {
            $doctores = [];
        }
        
        $this->set(compact('citas', 'usuario', 'doctores','coloresDoctores'));
        
        // Renderizar solo el element de la tabla
        $this->render('/element/tabla_citas');
        if ($this->request->is('ajax')) {
                $this->viewBuilder()->setLayout('ajax');
            } else {
                $this->viewBuilder()->setLayout('default');
            }
    }

}
