<?php
declare(strict_types=1);

namespace App\Controller;

/**
 * HorariosDoctores Controller
 *
 * @property \App\Model\Table\HorariosDoctoresTable $HorariosDoctores
 */
class HorariosDoctoresController extends AppController
{
    /**
     * Index method
     */
    public function index()
    {
        // Cargar doctores (usar variable local para evitar crear propiedad dinámica)
        $doctoresTable = $this->getTableLocator()->get('Doctores');
        $doctores = $doctoresTable->find(
            'list',
            keyField: 'id',
            valueField: function ($row) {
                return $row->nombre . ' ' . $row->apellido;
            }
        )->toArray();


        // Filtros
        $doctorId = $this->request->getQuery('doctor_id');
        $fecha = $this->request->getQuery('fecha');

        // Si no hay fecha, tomar hoy
        $fechaSeleccionada = $fecha ?: date('Y-m-d');

        // Crear rango de 7 días desde la fecha seleccionada
        $fechaInicio = date('Y-m-d', strtotime($fechaSeleccionada));
        $fechaFin = date('Y-m-d', strtotime($fechaSeleccionada . ' +6 days'));

        $horarios = [];

        if ($doctorId) {
            // Traer todos los horarios del doctor en el rango de 7 días
            $horarios = $this->HorariosDoctores->find()
                ->where([
                    'doctor_id' => $doctorId,
                    'fecha >=' => $fechaInicio,
                    'fecha <=' => $fechaFin,
                ])
                ->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])
                ->toArray();
        } 
        asort($doctores);
        // Variables para vista
        $this->set(compact('doctores', 'horarios', 'doctorId', 'fecha', 'fechaSeleccionada'));
    }

    /**
     * Bulk enable/delete slots for the next calendar month
     * POST params: doctor_id, action (enable|delete), hora_inicio, hora_fin, days[] (0-6 optional), skip_if_citas (0/1)
     */
    public function bulkModifyMonth()
    {
        $this->request->allowMethod(['post']);
        $this->autoRender = false;
        $this->response = $this->response->withType('application/json');

        $data = $this->request->getData();
        $doctorId = $data['doctor_id'] ?? null;
        $action = $data['action'] ?? 'enable';
        $horaInicio = $data['hora_inicio'] ?? null;
        $horaFin = $data['hora_fin'] ?? null;
    $days = $data['days'] ?? [];
    $skipIfCitas = !empty($data['skip_if_citas']);
    // Omitir la franja de descanso 13:00-15:00 si el cliente lo solicita
    $omitLunch = !empty($data['omit_lunch']);

        if (empty($doctorId) || empty($horaInicio) || empty($horaFin)) {
            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Parámetros incompletos']));
        }

        // Normalize times to H:i:s
        $horaInicio = date('H:i:s', strtotime($horaInicio));
        $horaFin = date('H:i:s', strtotime($horaFin));

        // compute next calendar month range
        $today = new \DateTimeImmutable('now');
        $firstDayNext = $today->modify('first day of next month')->setTime(0,0,0);
        $lastDayNext = $firstDayNext->modify('last day of this month')->setTime(0,0,0);
        $startDate = $firstDayNext->format('Y-m-d');
        $endDate = $lastDayNext->format('Y-m-d');

        // If days filter provided, normalize to ints
        $daysFilter = null;
        if (!empty($days) && is_array($days)) {
            $daysFilter = array_map('intval', $days);
        }

        $created = 0;
        $skipped = 0;
        $deleted = 0;

        $conn = $this->HorariosDoctores->getConnection();
        try {
            $conn->begin();

            if ($action === 'enable') {
                // fetch existing horarios in the range to avoid duplicates
                $existing = $this->HorariosDoctores->find()
                    ->select(['fecha', 'hora_inicio'])
                    ->where(['doctor_id' => $doctorId, 'fecha >=' => $startDate, 'fecha <=' => $endDate])
                    ->all();

                $existSet = [];
                foreach ($existing as $e) {
                    $existSet[$e->fecha->format('Y-m-d') . '_' . $e->hora_inicio->format('H:i:s')] = true;
                }

                $toSave = [];
                // iterate days
                $current = new \DateTime($startDate);
                $end = new \DateTime($endDate);
                while ($current <= $end) {
                    $w = (int)$current->format('w'); // 0 Sun .. 6 Sat
                    if ($daysFilter === null || in_array($w, $daysFilter, true)) {
                        // build slots between horaInicio and horaFin
                        $tsStart = strtotime($current->format('Y-m-d') . ' ' . $horaInicio);
                        $tsEnd = strtotime($current->format('Y-m-d') . ' ' . $horaFin);
                                for ($t = $tsStart; $t < $tsEnd; $t += 15*60) {
                                    // si se solicitó omitir el descanso, saltar franjas entre 13:00 y 15:00
                                    if ($omitLunch) {
                                        $lunchStart = strtotime($current->format('Y-m-d') . ' 13:00:00');
                                        $lunchEnd = strtotime($current->format('Y-m-d') . ' 15:00:00');
                                        if ($t >= $lunchStart && $t < $lunchEnd) {
                                            $skipped++;
                                            continue;
                                        }
                                    }
                            $hStart = date('H:i:s', $t);
                            $key = $current->format('Y-m-d') . '_' . $hStart;
                            if (isset($existSet[$key])) { $skipped++; continue; }
                            $toSave[] = [
                                'doctor_id' => $doctorId,
                                'fecha' => $current->format('Y-m-d'),
                                'hora_inicio' => $hStart,
                                'hora_fin' => date('H:i:s', $t + 15*60),
                                'dia_semana' => (int)$current->format('w'),
                            ];
                        }
                    }
                    $current->modify('+1 day');
                }

                if (!empty($toSave)) {
                    $entities = $this->HorariosDoctores->newEntities($toSave);
                    foreach ($entities as $e) {
                        if ($this->HorariosDoctores->save($e)) $created++; else $skipped++;
                    }
                }

                $conn->commit();
                return $this->response->withStringBody(json_encode(['success' => true, 'created' => $created, 'skipped' => $skipped]));
            }

            if ($action === 'delete') {
                // find candidate horarios in range matching time window and day filter
                $candidates = $this->HorariosDoctores->find()
                    ->where(['doctor_id' => $doctorId, 'fecha >=' => $startDate, 'fecha <=' => $endDate, 'hora_inicio >=' => $horaInicio, 'hora_inicio <' => $horaFin])
                    ->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])
                    ->all();

                $citasTable = $this->getTableLocator()->get('Citas');

                foreach ($candidates as $c) {
                    $w = (int)$c->fecha->format('w');
                    if ($daysFilter !== null && !in_array($w, $daysFilter, true)) continue;

                    // Si se solicitó omitir descanso, saltar slots entre 13:00 y 15:00
                    if ($omitLunch) {
                        $slotTs = strtotime($c->fecha->format('Y-m-d') . ' ' . $c->hora_inicio->format('H:i:s'));
                        $lunchStart = strtotime($c->fecha->format('Y-m-d') . ' 13:00:00');
                        $lunchEnd = strtotime($c->fecha->format('Y-m-d') . ' 15:00:00');
                        if ($slotTs >= $lunchStart && $slotTs < $lunchEnd) { $skipped++; continue; }
                    }

                    // if skipIfCitas: check if any cita exists at this slot (date + time)
                    if ($skipIfCitas) {
                        $pattern = $c->fecha->format('Y-m-d') . ' ' . substr($c->hora_inicio->format('H:i:s'),0,5) . '%';
                        $has = $citasTable->find()->where(['fecha_hora LIKE' => $pattern])->first();
                        if ($has) { $skipped++; continue; }
                    }

                    if ($this->HorariosDoctores->delete($c)) { $deleted++; } else { $skipped++; }
                }

                $conn->commit();
                return $this->response->withStringBody(json_encode(['success' => true, 'deleted' => $deleted, 'skipped' => $skipped]));
            }

            $conn->commit();
            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Acción desconocida']));

        } catch (\Throwable $e) {
            $conn->rollback();
            return $this->response->withStatus(500)->withStringBody(json_encode(['success' => false, 'message' => 'Error interno al procesar.']));
        }
    }

    /**
     * View method
     */
    public function view($id = null)
    {
        $doctor = $this->HorariosDoctores->Doctores->get($id, contain: ['HorariosDoctores']);
        $horariosDoctores = $this->HorariosDoctores->find()
            ->where(['doctor_id' => $id])
            ->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])
            ->all();

        $this->set(compact('doctor', 'horariosDoctores'));

        $layout = $this->request->is('ajax') ? 'ajax' : 'default';
        $this->viewBuilder()->setLayout($layout);
    }

    /**
     * Editar múltiples horarios de un doctor
     */
    public function editAll($doctorId)
    {
        $doctor = $this->HorariosDoctores->Doctores->get($doctorId);
        $horariosDoctores = $this->HorariosDoctores->find('all', [
            'conditions' => ['doctor_id' => $doctorId]
        ])->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])->toArray();

        if ($this->request->is(['patch', 'post', 'put'])) {
            $horariosData = $this->request->getData('horarios');

            foreach ($horariosDoctores as $index => $horario) {
                if (isset($horariosData[$index])) {
                    $horario = $this->HorariosDoctores->patchEntity($horario, $horariosData[$index]);
                    if (!$this->HorariosDoctores->save($horario)) {
                        $this->Flash->error(__('No se pudieron guardar algunos horarios.'));
                    }
                }
            }

            $this->Flash->success(__('Los horarios han sido actualizados.'));
            return $this->redirect(['action' => 'index']);
        }

        $this->set(compact('horariosDoctores', 'doctorId', 'doctor'));
        $layout = $this->request->is('ajax') ? 'ajax' : 'default';
        $this->viewBuilder()->setLayout($layout);
    }

    /**
     * Agregar horarios por fecha específica
     */
    public function add()
    {
        $doctores = $this->HorariosDoctores->Doctores->find('list', limit: 200)->all();

        if ($this->request->is('post')) {
            $data = $this->request->getData();
            $doctorId = $data['doctor_id'];
            $horarios = $data['horarios'] ?? [];
            $guardados = 0;

            foreach ($horarios as $horarioData) {
                $horarioEntity = $this->HorariosDoctores->newEmptyEntity();
                $horarioData['doctor_id'] = $doctorId;

                // Validar fecha específica
                if (empty($horarioData['fecha'])) {
                    continue; // saltar si no hay fecha
                }

                $this->HorariosDoctores->patchEntity($horarioEntity, $horarioData);

                if ($this->HorariosDoctores->save($horarioEntity)) {
                    $guardados++;
                }
            }

            if ($guardados > 0) {
                $this->Flash->success("Se guardaron {$guardados} horarios correctamente.");
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error('No se pudo guardar ningún horario. Revisa los datos.');
            }
        }

        $horariosDoctore = $this->HorariosDoctores->newEmptyEntity();
        $this->set(compact('horariosDoctore', 'doctores'));

        $layout = $this->request->is('ajax') ? 'ajax' : 'default';
        $this->viewBuilder()->setLayout($layout);
    }

    /**
     * Editar horario individual
     */
    public function edit($id = null)
    {
        $horariosDoctore = $this->HorariosDoctores->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $horariosDoctore = $this->HorariosDoctores->patchEntity($horariosDoctore, $this->request->getData());
            if ($this->HorariosDoctores->save($horariosDoctore)) {
                $this->Flash->success(__('El horario fue guardado correctamente.'));
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('El horario no se pudo guardar. Inténtalo nuevamente.'));
        }
        $doctores = $this->HorariosDoctores->Doctores->find('list', limit: 200)->all();
        $this->set(compact('horariosDoctore', 'doctores'));

        $layout = $this->request->is('ajax') ? 'ajax' : 'default';
        $this->viewBuilder()->setLayout($layout);
    }

    /**
     * Eliminar horario
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $horariosDoctore = $this->HorariosDoctores->get($id);
        if ($this->HorariosDoctores->delete($horariosDoctore)) {
            $this->Flash->success(__('El horario fue eliminado.'));
        } else {
            $this->Flash->error(__('El horario no se pudo eliminar. Inténtalo nuevamente.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    /**
     * Obtener horarios (usado por calendario)
     */
    public function obtenerHorarios()
    {
        $this->autoRender = false;
        $this->response = $this->response->withType('application/json');

        $doctor_id = $this->request->getQuery('doctor_id');

        if (!$doctor_id) {
            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Doctor ID no recibido']));
        }

        $horariosEntities = $this->HorariosDoctores->find()
            ->where(['doctor_id' => $doctor_id])
            ->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])
            ->toArray();

        if (empty($horariosEntities)) {
            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No hay horarios en la BD']));
        }

        $horarios = [];
        foreach ($horariosEntities as $h) {
            $horarios[] = [
                'id' => $h->id ?? null,
                'doctor_id' => $h->doctor_id ?? null,
                'fecha' => isset($h->fecha) && $h->fecha !== null ? $h->fecha->format('Y-m-d') : null,
                'dia_semana' => isset($h->dia_semana) ? (int)$h->dia_semana : null,
                'hora_inicio' => isset($h->hora_inicio) && $h->hora_inicio !== null ? $h->hora_inicio->format('H:i:s') : null,
                'hora_fin' => isset($h->hora_fin) && $h->hora_fin !== null ? $h->hora_fin->format('H:i:s') : null,
                'created' => isset($h->created) && $h->created !== null ? $h->created->format('Y-m-d H:i:s') : null,
                'modified' => isset($h->modified) && $h->modified !== null ? $h->modified->format('Y-m-d H:i:s') : null,
            ];
        }

        return $this->response->withStringBody(json_encode(['success' => true, 'horarios' => $horarios]));
    }
    
    /**
     * Devuelve el HTML del tbody del calendario para una fecha y doctor (uso AJAX)
     */
    public function partialTbody()
    {
        $this->request->allowMethod(['get']);
        $this->autoRender = false;

        $doctor_id = $this->request->getQuery('doctor_id');
        $fecha = $this->request->getQuery('fecha') ?: date('Y-m-d');

        if (!$doctor_id) {
            return $this->response->withStatus(400)->withStringBody('Missing doctor_id');
        }

        $horarios = $this->HorariosDoctores->find()
            ->where(['doctor_id' => $doctor_id, 'fecha >=' => $fecha, 'fecha <=' => date('Y-m-d', strtotime($fecha . ' +6 days'))])
            ->order(['fecha' => 'ASC', 'hora_inicio' => 'ASC'])
            ->toArray();

        // Render element parts and return JSON { thead, tbody }
        $view = $this->getViewBuilder()->setLayout(false);
        $thead = $view->setTemplatePath('element')->render('horarios_thead', ['fechaSeleccionada' => $fecha]);
        $tbody = $view->setTemplatePath('element')->render('horarios_tbody', ['fechaSeleccionada' => $fecha, 'horarios' => $horarios]);

        return $this->response->withType('application/json')->withStringBody(json_encode(['success' => true, 'thead' => $thead, 'tbody' => $tbody]));
    }
    // src/Controller/HorariosDoctoresController.php
public function asignarHorario()
{
    $this->request->allowMethod(['post']);
    $this->autoRender = false;
    $this->response = $this->response->withType('application/json');

    $data = $this->request->getData();

    // Validaciones básicas
    if (empty($data)) {
        return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No llegaron datos al servidor.']));
    }

    if (empty($data['doctor_id'])) {
        return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No se recibió el ID del doctor.']));
    }

    if (empty($data['fecha']) || empty($data['hora_inicio']) || empty($data['hora_fin'])) {
        return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Faltan los campos de fecha u hora.']));
    }

    // Normalizar formatos
    $data['fecha'] = date('Y-m-d', strtotime($data['fecha']));
    $data['hora_inicio'] = date('H:i:s', strtotime($data['hora_inicio']));
    $data['hora_fin'] = date('H:i:s', strtotime($data['hora_fin']));
    $data['dia_semana'] = (int)date('w', strtotime($data['fecha']));

    $accion = $data['accion'] ?? 'asignar';

    try {
        if ($accion === 'eliminar') {
            $horario = $this->HorariosDoctores->find()
                ->where([
                    'doctor_id' => $data['doctor_id'],
                    'fecha' => $data['fecha'],
                    'hora_inicio <=' => $data['hora_inicio'],
                    'hora_fin >' => $data['hora_inicio']
                ])
                ->first();

            if ($horario && $this->HorariosDoctores->delete($horario)) {
                return $this->response->withStringBody(json_encode(['success' => true, 'message' => 'Horario eliminado.']));
            }

            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No se pudo eliminar el horario.']));
        }

        if ($accion === 'mover') {
            // Se espera data.horario_id como id del horario origen
            $horarioId = $data['horario_id'] ?? null;
            if (empty($horarioId)) {
                return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No se recibió el ID del horario a mover.']));
            }

            $horario = $this->HorariosDoctores->get($horarioId);
            if (!$horario) {
                return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Horario origen no encontrado.']));
            }

            // Verificar que el destino no esté ocupado por otro horario del mismo doctor
            $conflicto = $this->HorariosDoctores->find()
                ->where([
                    'doctor_id' => $data['doctor_id'],
                    'fecha' => $data['fecha'],
                    'hora_inicio <' => $data['hora_fin'],
                    'hora_fin >' => $data['hora_inicio'],
                    'id !=' => $horarioId,
                ])
                ->first();

            if ($conflicto) {
                return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'El horario destino está ocupado.']));
            }

            // Actualizar horario con nueva fecha/hora
            $horario->fecha = $data['fecha'];
            $horario->hora_inicio = $data['hora_inicio'];
            $horario->hora_fin = $data['hora_fin'];
            $horario->dia_semana = $data['dia_semana'];

            if ($this->HorariosDoctores->save($horario)) {
                return $this->response->withStringBody(json_encode(['success' => true, 'message' => 'Horario movido correctamente.']));
            }

            return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'No se pudo mover el horario.']));
        }

        // Al asignar un rango, descomponer en franjas de 15 minutos
        if ($accion === 'asignar') {
            $startTs = strtotime($data['fecha'] . ' ' . $data['hora_inicio']);
            $endTs = strtotime($data['fecha'] . ' ' . $data['hora_fin']);
            if ($endTs <= $startTs) {
                return $this->response->withStringBody(json_encode(['success' => false, 'message' => 'Rango inválido: hora fin debe ser posterior a hora inicio.']));
            }

            $created = 0;
            $skipped = 0;

            for ($ts = $startTs; $ts < $endTs; $ts += 15 * 60) {
                $slotInicio = date('H:i:s', $ts);
                $slotFin = date('H:i:s', $ts + 15 * 60);

                // Verificar si ya existe un registro para doctor/fecha/hora_inicio
                $exists = $this->HorariosDoctores->find()
                    ->where([
                        'doctor_id' => $data['doctor_id'],
                        'fecha' => $data['fecha'],
                        'hora_inicio' => $slotInicio,
                    ])->first();

                if ($exists) {
                    $skipped++;
                    continue;
                }

                $slotData = [
                    'doctor_id' => $data['doctor_id'],
                    'fecha' => $data['fecha'],
                    'hora_inicio' => $slotInicio,
                    'hora_fin' => $slotFin,
                    'dia_semana' => $data['dia_semana'] ?? (int)date('w', strtotime($data['fecha'])),
                ];

                $entity = $this->HorariosDoctores->newEmptyEntity();
                $entity = $this->HorariosDoctores->patchEntity($entity, $slotData);
                if ($this->HorariosDoctores->save($entity)) {
                    $created++;
                } else {
                    // si falla guardar, contar como omitido
                    $skipped++;
                }
            }

            return $this->response->withStringBody(json_encode(['success' => true, 'message' => 'Rango procesado.', 'created' => $created, 'skipped' => $skipped]));
        }

    } catch (\Throwable $e) {
        return $this->response->withStatus(500)->withStringBody(json_encode(['success' => false, 'message' => 'Error interno del servidor.']));
    }
}

}
